#Requires -Version 5.1

<#
.SYNOPSIS
    Registers version information for Impact360 software installation.
.DESCRIPTION
    This script manages Windows registry entries for software installation/uninstallation.
    It reads version information from an XML file and handles version comparison for specific kits.
.PARAMETER InstallerSetupDir
    The directory containing the installer setup files.
.PARAMETER VersionInfoName
    The name of the version info XML file.
.EXAMPLE
    .\RegisterVersion.ps1 -InstallerSetupDir "C:\Install" -VersionInfoName "VersionInfo.xml"
.NOTES
    This script replaces the functionality of RegisterVersion.vbs
#>

param (
    [string]$InstallerSetupDir,
    [string]$VersionInfoName
)

# Constants
$SCRIPT_NAME = "RegisterVersion.ps1"
$ERROR_EXIT_CODE = -1
$SUCCESS_EXIT_CODE = 0

# Registry constants
$REG_KEY_UNINSTALL = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\{0}"
$REG_VALUE_DISPLAY_NAME = "DisplayName"
$REG_VALUE_DISPLAY_VERSION = "DisplayVersion"
$REG_VALUE_DISPLAY_PUBLISHER = "Publisher"
$REG_VALUE_DISPLAY_INSTALLDATE = "InstallDate"
$REG_VALUE_DISPLAY_UNINSTALLSTRING = "UninstallString"
$REG_VALUE_DISPLAY_NOREMOVE = "NoRemove"
$REG_VALUE_DISPLAY_NOMODIFY = "NoModify"

# Kit GUIDs
$GUID_APP_LIB_KIT = "{8833bafb-7c20-4025-8ab4-b3b010960caf}"
$GUID_COMPATIBILITY_KIT = "{85364735-0c6b-4133-9a1a-dfad80005488}"

# Function to write log messages with timestamp
function Write-LogMessage([string]$Message) {
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    Write-Host "[$timestamp] $Message"
}

# Function to validate script parameters
function Validate-ScriptParameters([string]$InstallerSetupDir, [string]$VersionInfoName) {
    if (-not $InstallerSetupDir) {
        throw "InstallerSetupDir parameter is required"
    }
    if (-not $VersionInfoName) {
        throw "VersionInfoName parameter is required"
    }
    
    $versionInfoPath = Join-Path $InstallerSetupDir $VersionInfoName
    if (-not (Test-Path $versionInfoPath)) {
        throw "Version info file not found: $versionInfoPath"
    }
}

# Function to get property value from XML
function Get-PropertyValue([xml]$XmlDoc, [string]$NodeName) {
    $node = $XmlDoc.SelectSingleNode("//$NodeName")
    if (-not $node) {
        Write-LogMessage "$NodeName not found in version info file"
        Write-LogMessage "$NodeName could not be identified."
        return "Unknown"
    }
    
    Write-LogMessage "$NodeName is '$($node.InnerText)'"
    return $node.InnerText
}

# Function to compare build numbers
function Compare-BuildNumbers([string]$Build1, [string]$Build2) {
    if (-not $Build1) {
        return -1
    }
    
    $segments1 = $Build1.Split('.')
    $segments2 = $Build2.Split('.')
    
    for ($i = 0; $i -lt 4; $i++) {
        if ([int]$segments1[$i] -lt [int]$segments2[$i]) {
            return -1
        }
        elseif ([int]$segments1[$i] -gt [int]$segments2[$i]) {
            return 1
        }
    }
    
    return 0
}

# Function to get install date in required format
function Get-InstallDate {
    $date = Get-Date
    return $date.ToString("yyyyMMdd")
}

# Main script execution
try {
    Write-LogMessage "Script $SCRIPT_NAME started"
    
    # Validate parameters
    Validate-ScriptParameters $InstallerSetupDir $VersionInfoName
    
    # Construct version info path
    $versionInfoPath = Join-Path $InstallerSetupDir $VersionInfoName
    Write-LogMessage "Opening $versionInfoPath"
    
    # Load and parse XML
    [xml]$xmlDoc = Get-Content -Path $versionInfoPath
    
    # Check if version registration is enabled
    $enabled = Get-PropertyValue $xmlDoc "Enabled"
    if ($enabled -ne "true") {
        Write-LogMessage "Adding version information for the package is disabled. Skipping Action."
        exit $SUCCESS_EXIT_CODE
    }
    Write-LogMessage "Adding version information for the package is enabled."
    
    # Get required properties
    $displayName = Get-PropertyValue $xmlDoc "DisplayName"
    $buildNumber = Get-PropertyValue $xmlDoc "BuildNumber"
    $guid = Get-PropertyValue $xmlDoc "guid"
    
    # Construct registry key path
    $uninstallKey = $REG_KEY_UNINSTALL -f $guid
    Write-LogMessage "Using registry key: $uninstallKey"
    
    # Check existing version
    Write-LogMessage "Checking existing DisplayVersion registry value"
    $installedBuildNumber = $null
    if (Test-Path $uninstallKey) {
        $installedBuildNumber = (Get-ItemProperty -Path $uninstallKey -Name $REG_VALUE_DISPLAY_VERSION -ErrorAction SilentlyContinue).$REG_VALUE_DISPLAY_VERSION
    }
    Write-LogMessage "Installed DisplayVersion: $installedBuildNumber"
    
    # Version comparison for specific kits
    if ($guid -in @($GUID_APP_LIB_KIT, $GUID_COMPATIBILITY_KIT)) {
        $res = Compare-BuildNumbers $installedBuildNumber $buildNumber
        if ($res -ge 0) {
            Write-LogMessage "Installed version is equal or newer. --> Skipping version registration."
            exit $SUCCESS_EXIT_CODE
        }
        Write-LogMessage "Kit version is greater than the installed version."
    }
    
    # Create or update registry values
    $regValues = @{
        $REG_VALUE_DISPLAY_NAME = $displayName
        $REG_VALUE_DISPLAY_VERSION = $buildNumber
        $REG_VALUE_DISPLAY_INSTALLDATE = (Get-InstallDate)
        $REG_VALUE_DISPLAY_PUBLISHER = "Impact360"
        $REG_VALUE_DISPLAY_UNINSTALLSTRING = " "
        $REG_VALUE_DISPLAY_NOREMOVE = 1
        $REG_VALUE_DISPLAY_NOMODIFY = 1
    }
    
    # Ensure registry key exists
    if (-not (Test-Path $uninstallKey)) {
        New-Item -Path $uninstallKey -Force | Out-Null
    }
    
    # Set registry values
    foreach ($regValue in $regValues.GetEnumerator()) {
        Write-LogMessage "Adding registry value $($regValue.Key) : $($regValue.Value)"
        if ($regValue.Key -in @($REG_VALUE_DISPLAY_NOREMOVE, $REG_VALUE_DISPLAY_NOMODIFY)) {
            Set-ItemProperty -Path $uninstallKey -Name $regValue.Key -Value $regValue.Value -Type DWord
        }
        else {
            Set-ItemProperty -Path $uninstallKey -Name $regValue.Key -Value $regValue.Value
        }
    }
    
    Write-LogMessage "Script $SCRIPT_NAME completed successfully"
    exit $SUCCESS_EXIT_CODE
}
catch {
    Write-LogMessage "FATAL ERROR: $($_.Exception.Message)"
    exit $ERROR_EXIT_CODE
} 
# SIG # Begin signature block
# MIIqzQYJKoZIhvcNAQcCoIIqvjCCKroCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBJ5XgTawIJiY+w
# meKKuDj4nccUIXSJUOKyuhyP2E80XKCCDiIwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggdqMIIFUqADAgECAhAGBCyY6obPZZn6S1XIjHwXMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQwNTEwMDAwMDAwWhcNMjcwNTEy
# MjM1OTU5WjByMQswCQYDVQQGEwJVUzEQMA4GA1UECBMHR2VvcmdpYTETMBEGA1UE
# BxMKQWxwaGFyZXR0YTEdMBsGA1UEChMUVkVSSU5UIEFNRVJJQ0FTIElOQy4xHTAb
# BgNVBAMTFFZFUklOVCBBTUVSSUNBUyBJTkMuMIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAtaB6Abe0BiBJ9bmAr9uNBZcdoj4Jy28w9mn3htPK6wpks1pm
# T25/Trm3WYOeK0XRiIFj1xVHpWwNuouAFZqeVAk4P2YwmY5qvRAM8ZybL5CUF3Wt
# /rygDrrixDth4Oz3HqtN2ERGhsBQKWgYQCehRUJ41tZBehPgGmWpxt0JfruWKrV7
# UDVioeO91Rm3YwBqUayrLwxLznGOEQK5xp4KgglzXXUtbXlpW3UCUh8ZctKQ9Lao
# 5zxGKyynuwTSnQ70uAhu7R0WhUWfbgwtKo7/iGi4wZMrgsBwlnJ4tCHKtmzdavn5
# SAR5mvR8OoTRQ8V3yET9EZCMkrFL/NC9sdU/MqBxnAL2LRhU1Iqa1+RbfE+qw0Aj
# kIT6CvUu+mClG/XPTIk0jgaZ5hSYDBb6z3njVdNhTCONCr04RqIDP9A3ba5i1wPn
# FNpe2TM83bz3xMntTNsgdNevXrMIRgUDU0svopOBaymKghce+GVaEfMITxGtOZls
# SMRKnW+UHOPS8Kq5nzWNJer8c36fwr5T3Tor8l6qTraPFB/k557/PixckLzQWC5V
# vJtKuDdkrlX3NCNVYy9Hb5iZ4nCM/nBv6if8gGX4oW5ipTI0TiTc1+W5mzgex8Sv
# 2lkASHM1KMI6FzqTJfTb7pNgcALDnH/12Lv1BWPZvg5UtLOPJkNsxkeSuPkCAwEA
# AaOCAgMwggH/MB8GA1UdIwQYMBaAFGg34Ou2O/hfEYb7/mF7CIhl9E5CMB0GA1Ud
# DgQWBBRq+jHGGFc/ue8kaN00LNeriqN5DTA+BgNVHSAENzA1MDMGBmeBDAEEATAp
# MCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0P
# AQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0wgaowU6BR
# oE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENv
# ZGVTaWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRwOi8v
# Y3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JT
# QTQwOTZTSEEzODQyMDIxQ0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYB
# BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBcBggrBgEFBQcwAoZQaHR0
# cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNp
# Z25pbmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0TBAIwADANBgkqhkiG
# 9w0BAQsFAAOCAgEARWLJkxq9VhUiL6TpjnNkAg9jU4Vw+VmfpTkIyWL7malbUP0Y
# rfI3PspRaS1m2KIp+SEebUR4sqjlDUFAyl0BeO8wse1WAMyze9FInocaGSTNmlJu
# Q5lgDlbJJ6CvdTgcFMRDKuUw5oZJsWG7M2AZyOZoHPPg6l84giH+MtXfXlu5+BCV
# 8bVRdblnRYtz4W5WaQLFqDBf24CZQ3z7hqsWnm2WZgkv1Y22WU8Q931XjjIpYI4q
# plp8VlWJ0ugcoQdjWzLWmiTMOHK1svaYIgZub11Se0eUaffw/gOOT71bPvysC47I
# mJL7QnUTznrMDiddvYy+2WTA7bdaA2ev3U8QLNr3Esr0gLYi7ZO9A0iBGIF/M0CZ
# TqYcpMRRoEJUe+Pcwsyq1Ygl/REkGN0kxMxdYlEX6rwDJMi+lwILNb96ZXImRgIw
# spsJz6bcGZo50aX9hTfmT5btKOkW5bnJo0ximax1wraJTpQ5KMPS7OC0blDJk4y1
# O9hVnpWf/XpikcvoCe++gBhDh88IlYGBq+upIVO1IQ57ovJPfFXO/QPZls/m28xz
# DsDKTXgATrAQ1iMmcSB59OYWmsViUdunEOUoOMB1zaOXy/qrN9ceAdrq6yMuGIjW
# 3eJqhgeWKqozdjasYzzFiXSqnNLEMxkPel+99ps74CKeYBdAEXRKmkUFkJ8xghwB
# MIIb/QIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5j
# LjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcgUlNB
# NDA5NiBTSEEzODQgMjAyMSBDQTECEAYELJjqhs9lmfpLVciMfBcwDQYJYIZIAWUD
# BAIBBQCgfDAQBgorBgEEAYI3AgEMMQIwADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGC
# NwIBBDAcBgorBgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQx
# IgQgxEFgpLQv3Q2J7pXm7QXFyGUeSKqIa2BOJSf9ud4mER0wDQYJKoZIhvcNAQEB
# BQAEggIAc9Tvb6AlqMBwoLyJ1YZiuMQKAL73z23pcT0Mb9WdfNZxSmMC6YYEeLEx
# OhSt2/7bqvtk2x3A8mRr/Nsn660AakJ+9P4VBfjTVREXMUGH8ONcPW1YA4JFjM/n
# K9QCXnGilzRHRyRH7kqw7N8zVswYdbKs2pCKcWzfbE/g4DY/DLQ6c0QK1D2dYvGQ
# 7ejHSc0+cjA9/IO6NSh47Mu4HiuSULDopYbd5iGsE7fo5Df5MuqiQcBok4171W8N
# wPaWu3OWhiTBi7+TNg28GmNd7z3OCbDR8vRB2jbDuHsO6xkkxWKhV80SJ5SdKuYh
# FZSHpslV+5j7taxaBONeMbdCwmWoAqkChRCzXFe2EWCCARTpeQoLGx0InREFkwVl
# k0svIz2Wl1pmO74j8fU+PDi/AFtUQDxsNgInGKoRa7HGkGp7+cFf0JBf4yGarczB
# KhfMDxz3bEK9agWmZCZMm8TsV+letc19HWAcTo4f1EKCTZukg+Nph8+CLVrpkUb7
# +iXvTHG8aHACO+uIMA4rjcn7O+dOFpoM6klO5FT2rLzkVHaPNZdQs6Pn8A65XfqP
# nm0V1oMGC+yA0EuBqRdFN5j7vCkqaYzUlGCpGM5A2bi2VDncjv6M6Ie8vWzgRSCP
# SxpRgYPCrte9p0MvU/zH5TG6fI6msJ3NkyUzRq6q3coH1Vo17mChghjXMIIY0wYK
# KwYBBAGCNwMDATGCGMMwghi/BgkqhkiG9w0BBwKgghiwMIIYrAIBAzEPMA0GCWCG
# SAFlAwQCAgUAMIH3BgsqhkiG9w0BCRABBKCB5wSB5DCB4QIBAQYKKwYBBAGyMQIB
# ATAxMA0GCWCGSAFlAwQCAQUABCBCcsgd7aSI0ACX5/FXCxPTxsQ1uF5n685YCG8s
# ry/lTgIUeOsaO4lf66pr7lJ6UZA88GN6LqIYDzIwMjUwNzE1MTMwNTI4WqB2pHQw
# cjELMAkGA1UEBhMCR0IxFzAVBgNVBAgTDldlc3QgWW9ya3NoaXJlMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUg
# U3RhbXBpbmcgU2lnbmVyIFIzNqCCEwQwggZiMIIEyqADAgECAhEApCk7bh7d16c0
# CIetek63JDANBgkqhkiG9w0BAQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMP
# U2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0
# YW1waW5nIENBIFIzNjAeFw0yNTAzMjcwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMHIx
# CzAJBgNVBAYTAkdCMRcwFQYDVQQIEw5XZXN0IFlvcmtzaGlyZTEYMBYGA1UEChMP
# U2VjdGlnbyBMaW1pdGVkMTAwLgYDVQQDEydTZWN0aWdvIFB1YmxpYyBUaW1lIFN0
# YW1waW5nIFNpZ25lciBSMzYwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQDThJX0bqRTePI9EEt4Egc83JSBU2dhrJ+wY7JgReuff5KQNhMuzVytzD+iXazA
# TVPMHZpH/kkiMo1/vlAGFrYN2P7g0Q8oPEcR3h0SftFNYxxMh+bj3ZNbbYjwt8f4
# DsSHPT+xp9zoFuw0HOMdO3sWeA1+F8mhg6uS6BJpPwXQjNSHpVTCgd1gOmKWf12H
# SfSbnjl3kDm0kP3aIUAhsodBYZsJA1imWqkAVqwcGfvs6pbfs/0GE4BJ2aOnciKN
# iIV1wDRZAh7rS/O+uTQcb6JVzBVmPP63k5xcZNzGo4DOTV+sM1nVrDycWEYS8bSS
# 0lCSeclkTcPjQah9Xs7xbOBoCdmahSfg8Km8ffq8PhdoAXYKOI+wlaJj+PbEuwm6
# rHcm24jhqQfQyYbOUFTKWFe901VdyMC4gRwRAq04FH2VTjBdCkhKts5Py7H73obM
# GrxN1uGgVyZho4FkqXA8/uk6nkzPH9QyHIED3c9CGIJ098hU4Ig2xRjhTbengonc
# XUeo/cfpKXDeUcAKcuKUYRNdGDlf8WnwbyqUblj4zj1kQZSnZud5EtmjIdPLKce8
# UhKl5+EEJXQp1Fkc9y5Ivk4AZacGMCVG0e+wwGsjcAADRO7Wga89r/jJ56IDK773
# LdIsL3yANVvJKdeeS6OOEiH6hpq2yT+jJ/lHa9zEdqFqMwIDAQABo4IBjjCCAYow
# HwYDVR0jBBgwFoAUX1jtTDF6omFCjVKAurNhlxmiMpswHQYDVR0OBBYEFIhhjKEq
# N2SBKGChmzHQjP0sAs5PMA4GA1UdDwEB/wQEAwIGwDAMBgNVHRMBAf8EAjAAMBYG
# A1UdJQEB/wQMMAoGCCsGAQUFBwMIMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMI
# MCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEE
# AjBKBgNVHR8EQzBBMD+gPaA7hjlodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3Rp
# Z29QdWJsaWNUaW1lU3RhbXBpbmdDQVIzNi5jcmwwegYIKwYBBQUHAQEEbjBsMEUG
# CCsGAQUFBzAChjlodHRwOi8vY3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNU
# aW1lU3RhbXBpbmdDQVIzNi5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4IBgQACgT6khnJRIfllqS49Uorh5ZvM
# SxNEk4SNsi7qvu+bNdcuknHgXIaZyqcVmhrV3PHcmtQKt0blv/8t8DE4bL0+H0m2
# tgKElpUeu6wOH02BjCIYM6HLInbNHLf6R2qHC1SUsJ02MWNqRNIT6GQL0Xm3LW7E
# 6hDZmR8jlYzhZcDdkdw0cHhXjbOLsmTeS0SeRJ1WJXEzqt25dbSOaaK7vVmkEVkO
# Hsp16ez49Bc+Ayq/Oh2BAkSTFog43ldEKgHEDBbCIyba2E8O5lPNan+BQXOLuLMK
# YS3ikTcp/Qw63dxyDCfgqXYUhxBpXnmeSO/WA4NwdwP35lWNhmjIpNVZvhWoxDL+
# PxDdpph3+M5DroWGTc1ZuDa1iXmOFAK4iwTnlWDg3QNRsRa9cnG3FBBpVHnHOEQj
# 4GMkrOHdNDTbonEeGvZ+4nSZXrwCW4Wv2qyGDBLlKk3kUW1pIScDCpm/chL6aUbn
# SsrtbepdtbCLiGanKVR/KC1gsR0tC6Q0RfWOI4owggYUMIID/KADAgECAhB6I67a
# U2mWD5HIPlz0x+M/MA0GCSqGSIb3DQEBDAUAMFcxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIFRp
# bWUgU3RhbXBpbmcgUm9vdCBSNDYwHhcNMjEwMzIyMDAwMDAwWhcNMzYwMzIxMjM1
# OTU5WjBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSww
# KgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjCCAaIw
# DQYJKoZIhvcNAQEBBQADggGPADCCAYoCggGBAM2Y2ENBq26CK+z2M34mNOSJjNPv
# IhKAVD7vJq+MDoGD46IiM+b83+3ecLvBhStSVjeYXIjfa3ajoW3cS3ElcJzkyZlB
# nwDEJuHlzpbN4kMH2qRBVrjrGJgSlzzUqcGQBaCxpectRGhhnOSwcjPMI3G0hedv
# 2eNmGiUbD12OeORN0ADzdpsQ4dDi6M4YhoGE9cbY11XxM2AVZn0GiOUC9+XE0wI7
# CQKfOUfigLDn7i/WeyxZ43XLj5GVo7LDBExSLnh+va8WxTlA+uBvq1KO8RSHUQLg
# zb1gbL9Ihgzxmkdp2ZWNuLc+XyEmJNbD2OIIq/fWlwBp6KNL19zpHsODLIsgZ+WZ
# 1AzCs1HEK6VWrxmnKyJJg2Lv23DlEdZlQSGdF+z+Gyn9/CRezKe7WNyxRf4e4bwU
# trYE2F5Q+05yDD68clwnweckKtxRaF0VzN/w76kOLIaFVhf5sMM/caEZLtOYqYad
# tn034ykSFaZuIBU9uCSrKRKTPJhWvXk4CllgrwIDAQABo4IBXDCCAVgwHwYDVR0j
# BBgwFoAU9ndq3T/9ARP/FqFsggIv0Ao9FCUwHQYDVR0OBBYEFF9Y7UwxeqJhQo1S
# gLqzYZcZojKbMA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMG
# A1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBMBgNVHR8ERTBD
# MEGgP6A9hjtodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1l
# U3RhbXBpbmdSb290UjQ2LmNybDB8BggrBgEFBQcBAQRwMG4wRwYIKwYBBQUHMAKG
# O2h0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGlu
# Z1Jvb3RSNDYucDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNv
# bTANBgkqhkiG9w0BAQwFAAOCAgEAEtd7IK0ONVgMnoEdJVj9TC1ndK/HYiYh9lVU
# acahRoZ2W2hfiEOyQExnHk1jkvpIJzAMxmEc6ZvIyHI5UkPCbXKspioYMdbOnBWQ
# Un733qMooBfIghpR/klUqNxx6/fDXqY0hSU1OSkkSivt51UlmJElUICZYBodzD3M
# /SFjeCP59anwxs6hwj1mfvzG+b1coYGnqsSz2wSKr+nDO+Db8qNcTbJZRAiSazr7
# KyUJGo1c+MScGfG5QHV+bps8BX5Oyv9Ct36Y4Il6ajTqV2ifikkVtB3RNBUgwu/m
# SiSUice/Jp/q8BMk/gN8+0rNIE+QqU63JoVMCMPY2752LmESsRVVoypJVt8/N3qQ
# 1c6FibbcRabo3azZkcIdWGVSAdoLgAIxEKBeNh9AQO1gQrnh1TA8ldXuJzPSuALO
# z1Ujb0PCyNVkWk7hkhVHfcvBfI8NtgWQupiaAeNHe0pWSGH2opXZYKYG4Lbukg7H
# pNi/KqJhue2Keak6qH9A8CeEOB7Eob0Zf+fU+CCQaL0cJqlmnx9HCDxF+3BLbUuf
# rV64EbTI40zqegPZdA+sXCmbcZy6okx/SjwsusWRItFA3DE8MORZeFb6BmzBtqKJ
# 7l939bbKBy2jvxcJI98Va95Q5JnlKor3m0E7xpMeYRriWklUPsetMSf2NvUQa/E5
# vVyefQIwggaCMIIEaqADAgECAhA2wrC9fBs656Oz3TbLyXVoMA0GCSqGSIb3DQEB
# DAUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UE
# BxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEu
# MCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAe
# Fw0yMTAzMjIwMDAwMDBaFw0zODAxMTgyMzU5NTlaMFcxCzAJBgNVBAYTAkdCMRgw
# FgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGlj
# IFRpbWUgU3RhbXBpbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQCIndi5RWedHd3ouSaBmlRUwHxJBZvMWhUP2ZQQRLRBQIF3FJmp1OR2
# LMgIU14g0JIlL6VXWKmdbmKGRDILRxEtZdQnOh2qmcxGzjqemIk8et8sE6J+N+Gl
# 1cnZocew8eCAawKLu4TRrCoqCAT8uRjDeypoGJrruH/drCio28aqIVEn45NZiZQI
# 7YYBex48eL78lQ0BrHeSmqy1uXe9xN04aG0pKG9ki+PC6VEfzutu6Q3IcZZfm00r
# 9YAEp/4aeiLhyaKxLuhKKaAdQjRaf/h6U13jQEV1JnUTCm511n5avv4N+jSVwd+W
# b8UMOs4netapq5Q/yGyiQOgjsP/JRUj0MAT9YrcmXcLgsrAimfWY3MzKm1HCxcqu
# inTqbs1Q0d2VMMQyi9cAgMYC9jKc+3mW62/yVl4jnDcw6ULJsBkOkrcPLUwqj7po
# S0T2+2JMzPP+jZ1h90/QpZnBkhdtixMiWDVgh60KmLmzXiqJc6lGwqoUqpq/1HVH
# m+Pc2B6+wCy/GwCcjw5rmzajLbmqGygEgaj/OLoanEWP6Y52Hflef3XLvYnhEY4k
# SirMQhtberRvaI+5YsD3XVxHGBjlIli5u+NrLedIxsE88WzKXqZjj9Zi5ybJL2Wj
# eXuOTbswB7XjkZbErg7ebeAQUQiS/uRGZ58NHs57ZPUfECcgJC+v2wIDAQABo4IB
# FjCCARIwHwYDVR0jBBgwFoAUU3m/WqorSs9UgOHYm8Cd8rIDZsswHQYDVR0OBBYE
# FPZ3at0//QET/xahbIICL9AKPRQlMA4GA1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8E
# BTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBQ
# BgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5jb20vVVNFUlRy
# dXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwNQYIKwYBBQUHAQEEKTAn
# MCUGCCsGAQUFBzABhhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3
# DQEBDAUAA4ICAQAOvmVB7WhEuOWhxdQRh+S3OyWM637ayBeR7djxQ8SihTnLf2sA
# BFoB0DFR6JfWS0snf6WDG2gtCGflwVvcYXZJJlFfym1Doi+4PfDP8s0cqlDmdfyG
# OwMtGGzJ4iImyaz3IBae91g50QyrVbrUoT0mUGQHbRcF57olpfHhQEStz5i6hJvV
# LFV/ueQ21SM99zG4W2tB1ExGL98idX8ChsTwbD/zIExAopoe3l6JrzJtPxj8V9ro
# cAnLP2C8Q5wXVVZcbw4x4ztXLsGzqZIiRh5i111TW7HV1AtsQa6vXy633vCAbAOI
# aKcLAo/IU7sClyZUk62XD0VUnHD+YvVNvIGezjM6CRpcWed/ODiptK+evDKPU2K6
# synimYBaNH49v9Ih24+eYXNtI38byt5kIvh+8aW88WThRpv8lUJKaPn37+YHYafo
# b9Rg7LyTrSYpyZoBmwRWSE4W6iPjB7wJjJpH29308ZkpKKdpkiS9WNsf/eeUtvRr
# tIEiSJHN899L1P4l6zKVsdrUu1FX1T/ubSrsxrYJD+3f3aKg6yxdbugot06YwGXX
# iy5UUGZvOu3lXlxA+fC13dQ5OlL2gIb5lmF6Ii8+CQOYDwXM+yd9dbmocQsHjcRP
# sccUd5E9FiswEqORvz8g3s+jR3SFCgXhN4wz7NgAnOgpCdUo4uDyllU9PzGCBJIw
# ggSOAgEBMGowVTELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDEsMCoGA1UEAxMjU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYC
# EQCkKTtuHt3XpzQIh616TrckMA0GCWCGSAFlAwQCAgUAoIIB+TAaBgkqhkiG9w0B
# CQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI1MDcxNTEzMDUyOFow
# PwYJKoZIhvcNAQkEMTIEMDNsJZ9JVKTwO3EGr9me6o2NO1i8ZT85Z63dsjS3k2M1
# sfMFZhviUSyFn2l4XTMGbDCCAXoGCyqGSIb3DQEJEAIMMYIBaTCCAWUwggFhMBYE
# FDjJFIEQRLTcZj6T1HRLgUGGqbWxMIGHBBTGrlTkeIbxfD1VEkiMacNKevnC3TBv
# MFukWTBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4w
# LAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2AhB6
# I67aU2mWD5HIPlz0x+M/MIG8BBSFPWMtk4KCYXzQkDXEkd6SwULaxzCBozCBjqSB
# izCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcT
# C0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAs
# BgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDbC
# sL18Gzrno7PdNsvJdWgwDQYJKoZIhvcNAQEBBQAEggIALXZqRcPqa4CyK+Hp90G0
# 8ZBXupMYIC7jPWhtN1jHZ7hjV+VAaqOtqNQgP+SnDFKzW4aa+CLRKFJc8auxKz9C
# wxKDAI/I4yGl22tYmZLi9WI1VxhC2RFKhK9CgGIbxwU1RerpKL5AYsEdxne4Vz6M
# AEduubYxTd9gpK0bcE/feLo3PCFAAxWXRobMikib2RgsB0gshP+JERIK777be3eH
# EzAq0s9qG/+RAWlproY38NBGetsGNuPwFUlRjXJZcM1wi/KcgRldXLsxTbrX0vL0
# SwUYZ0wYbW7mu0Qrp8AoeU8RX4sJ0jv1oS8MFue3OHfx4JKNrfNfwAigkrWt4LML
# HJcUOlVFkNhiy57hK5hJuH6LlwziTq0ZiFeMwjCJUxQ574GL/xa5JBPjdym+wS6N
# 2MiiZyuTKSaVQWSAZHyKXZH6Ea6J3rnKp4e82OoJKmc0a0TPfmppiBAbQZ/KKlZa
# Q9WtTWBGG1yGE4F7lcN5xoNLU4Hiz+vN9MtQc0+RMzsKnKHZO6NGTc4kXAUNZzKX
# loqPQn9CloIUqE97DULtpyeGpALAd0Gari2/qIMaBm6ibTn+Ml92LZcoVJEiOOse
# BkP6q8lN7WuO4/EU7fpbkGPvm6P5fIDauIEyzzzX8ngtVD7AjIo/ZwXY1nV+MPqC
# 2pXeYjsow/z3MwOF6U/ViBE=
# SIG # End signature block
